//
// display.cpp
//
// Circle - A C++ bare metal environment for Raspberry Pi
// Copyright (C) 2015  R. Stange <rsta2@o2online.de>
// 
// PiTap (C) 2025 Mike Dawson https://gp2x.org/pitap
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//


#include <circle/screen.h>
#include <string.h>

#include "display.h"

Display *Display::s_pThis = 0;
//CDisplay *pCDisplay;

CString currentDir;
//FILINFO *dirList;
int dirNum;
int cursor;
//int textHeight=16;
//int text_y;
boolean dirChanged=FALSE;
boolean fileChanged=FALSE;
boolean recordStopped=FALSE;
boolean doScreenshot=FALSE;
boolean updateLCD=TRUE;

const char *lastKey;
//boolean mShowCursor=FALSE;
//unsigned mPosX;
//unsigned mPosY;
//unsigned mButtons;
//unsigned tPosX;
//unsigned tPosY;

//#define SCREEN_WIDTH 1280
//#define SCREEN_HEIGHT 720
//#define BORDER_WIDTH   64
//#define BORDER_HEIGHT  64
//#define BROWSER_HEIGHT 22

//u64 lastPulses[SCREEN_HEIGHT];

Display::Display (CDeviceNameService *devicenameservice, File *pFile, FS *pFS, Tap *pTap, TapeCart *pTapeCart, Status *pStatus, CTimer *pTimer, LCD *pLCD)
	:	m_pDeviceNameService (devicenameservice),
		m_pKeyb (0),
		m_pMouse (0),
		m_pTouchScreen (0),
		m_fb (SCREEN_WIDTH, SCREEN_HEIGHT),
		m_pTimer (pTimer),
		m_pFile (pFile),
		m_pFS (pFS),
		m_pTap (pTap),
		m_pTapeCart (pTapeCart),
		m_pStatus (pStatus),
		m_pLCD (pLCD)
{
	s_pThis = this;
}

Display::~Display (void)
{
	s_pThis = 0;
}

void Display::drawDir ()
{
	int i;
	CString name;
	int start=0;
	if(cursor>(BROWSER_HEIGHT/2)) start=cursor-(BROWSER_HEIGHT/2);
	text_y+=textHeight;
	T2DColor textColour;

	if(updateLCD)
	{
		m_pLCD->setContent(2, "", FALSE);
		m_pLCD->setContent(3, "", FALSE);
		m_pLCD->setContent(4, "", FALSE);
	}

	for(i=start; i<(start+BROWSER_HEIGHT) && i<dirNum; i++)
	{
		if(i==cursor) textColour=COLOR_SELECT;
		else textColour=COLOR_FG;
		if(dirList[i].fattrib&AM_DIR) name.Format("[DIR] %s", dirList[i].fname);
		else name.Format("      %s", dirList[i].fname);
		m_fb.DrawText(100, text_y+=textHeight, textColour, name);

		if(updateLCD && i==cursor-1)
		{
			name.Format("%s", dirList[i].fname);
			m_pLCD->setContent(2, name, FALSE);
		}
		if(updateLCD && i==cursor)
		{
			name.Format("%s", dirList[i].fname);
			m_pLCD->setContent(3, name, TRUE);
		}
		if(updateLCD && i==cursor+1)
		{
			name.Format("%s", dirList[i].fname);
			m_pLCD->setContent(4, name, FALSE);
		}
	}
	updateLCD=FALSE;
}

void Display::checkButtons()
{
	if(m_pLCD->buttonOn[0])	// select
	{
		selectDirEntry();
		m_pLCD->buttonOn[0]=FALSE;
	}
	if(m_pLCD->buttonOn[1]) // up
	{
		if(cursor>0)
		{
			cursor--;
			updateLCD=TRUE;
		}
		m_pLCD->buttonOn[1]=FALSE;
	}
	if(m_pLCD->buttonOn[2]) // down
	{
		if(cursor<(dirNum-1))
		{
			cursor++;
			updateLCD=TRUE;
		}
		m_pLCD->buttonOn[2]=FALSE;
	}
	if(m_pLCD->buttonOn[3]) // exit/back
	{
		navBack();
		m_pLCD->buttonOn[3]=FALSE;
	}
	if(m_pLCD->buttonOn[4]) // 5 long-press - mount /pitap/browser.prg
	{
		u64 timeNow=m_pTimer->GetClockTicks64();
		if(m_pLCD->buttonPressTime[4] && (timeNow-m_pLCD->buttonPressTime[4]>1500000))
		{
			m_pStatus->currentTap="/pitap/browser.prg";
			fileChanged=TRUE;
			m_pLCD->buttonOn[4]=FALSE;
		}
	}
	if(m_pLCD->buttonDepress[4]) // 5 press - play/stop
	{
		if(!m_pTap->isPlaying())
		{
			m_pTap->playTap();
			//m_pTapeCart->setMode(MODE_STREAM);
		} else {
			if(m_pTap->isRecording())
			{
				m_pTap->stopTap();
				recordStopped=TRUE;
			} else {
				m_pTap->stopTap();
			}
		}
		m_pLCD->buttonDepress[4]=FALSE;
	}
}

void Display::drawScreen ()
{
	static unsigned frameCount=0;
	static double fps;
	static u64 lastUpdate;
	static int pulseWrite=0, pulseRead=0;

	if(m_pTouchScreen!=0) m_pTouchScreen->Update();

	checkButtons();
	
	if(dirChanged)
	{
		updateDir();
		dirChanged=FALSE;
		updateLCD=TRUE;
	}
	if(fileChanged)
	{
		m_pFS->mountFile(m_pStatus->currentTap);
		fileChanged=FALSE;
		updateLCD=TRUE;
	}
	if(recordStopped)
	{
		unsigned size=m_pTap->getTapSize();
		//m_pFS->writeFile(m_pStatus->tap_buffer, size, "record.tap");
		m_pFile->writeFile(m_pStatus->tap_buffer, size, "record.tap");
		m_pStatus->tap_length=m_pTap->getTapLength();
		recordStopped=FALSE;
	}

	u32 *screenBuf=(u32 *)m_fb.GetBuffer();
	CString s;
	text_y=64;
	T2DColor textColour;
	textColour=COLOR_FG;

	int pulseLength=m_pTap->getLastPulseLength();
	for(int i=0; i<(pulseLength/8) && i<SCREEN_HEIGHT; i++) lastPulses[(pulseWrite++)%SCREEN_HEIGHT]=pulseLength;

	u32 bColor;
	pulseRead+=(SCREEN_HEIGHT/4);
	for(int y=0; y<SCREEN_HEIGHT; y++)
	{
		int pulse=lastPulses[(pulseRead++)%SCREEN_HEIGHT];
		bColor=COLOR64_LBLUE;
		//if(m_pTap->getMotor()) {
		if(m_pTap->isBusy()) {
			bColor=COLOR64_BLACK;
			if(pulse>100) bColor=COLOR64_GREY1;
			if(pulse>200) bColor=COLOR64_GREY2;
			if(pulse>250) bColor=COLOR64_GREY3;
			if(pulse>300) bColor=COLOR64_WHITE;
			if(pulse>400) bColor=COLOR64_RED;
			if(pulse>500) bColor=COLOR64_BLUE;
			if(pulse>700) bColor=COLOR64_VIOLET;
		}

		for(int x=0; x<SCREEN_WIDTH; x++)
		{
			if(y<BORDER_HEIGHT || y>(SCREEN_HEIGHT-BORDER_HEIGHT) || x<BORDER_WIDTH || x>(SCREEN_WIDTH-BORDER_WIDTH))
			{
				screenBuf[(y*SCREEN_WIDTH)+x]=bColor;
			} else {
				screenBuf[(y*SCREEN_WIDTH)+x]=COLOR_BG;
			}
		}
	}

	s.Format("PiTap v" PITAP_VER);
	m_fb.DrawText(100, text_y+=textHeight, textColour, s);

	s.Format("Web interface: http://%s\n", (const char *)m_pStatus->ipAddress);
	m_fb.DrawText(100, text_y+=textHeight, textColour, s);

	s.Format("F1 Play, F2 Stop, F4 Record, F5 Browse, F8 Reboot, +/- Ffwd/Rew \n");
	m_fb.DrawText(100, text_y+=textHeight, textColour, s);

	s.Format("Mounted tap file: %s\n", (const char *)m_pStatus->currentTap);
	m_fb.DrawText(100, text_y+=textHeight, textColour, s);

	s.Format("Counter: %d\n", m_pTap->getCounterSeconds());
	m_fb.DrawText(100, text_y+=textHeight, textColour, s);

	const char *stat;
	if(m_pTapeCart->tapecartMode!=MODE_STREAM) stat="tapecart";
	else if(m_pTap->isPlaying()) stat="playing";
	else if(m_pTap->isRecording()) stat="recording";
	else stat="stopped";
	s.Format("Status: %s\n", stat);
	m_fb.DrawText(100, text_y+=textHeight, textColour, s);

	s.Format("Temperature: %2d, clock: %3dmhz, FPS: %2.2f\n", m_pStatus->cpuTemp, m_pStatus->clockRate/1000000, fps);
	m_fb.DrawText(100, text_y+=textHeight, textColour, s);

	//s.Format("Last key: %s", (const char *)lastKey);
	//m_fb.DrawText(100, text_y+=textHeight, textColour, s);
	//s.Format("Mouse: %d,%d %x %x", mPosX, mPosY, mButtons);
	//m_fb.DrawText(100, text_y+=textHeight, textColour, s);
	//s.Format("Touch screen: %d,%d", tPosX, tPosY);
	//m_fb.DrawText(100, text_y+=textHeight, textColour, s);

	s.Format("Directory: %s", (const char *)currentDir);
	text_y+=textHeight;
	m_fb.DrawText(100, text_y+=textHeight, textColour, s);
	if(dirNum>0) drawDir();

	if(m_pStatus->img_data!=0)
	{
		int startx=832;
		int starty=128;
		int imgWidth=m_pStatus->img_width;
		int imgHeight=m_pStatus->img_height;
		int dWidth=320;
		int dHeight=200;

		for(int y=0; y<dHeight; y++)
		{
			for(int x=0; x<dWidth; x++)
			{
				int dx=((y*(imgHeight/dHeight))*imgWidth)+(x*imgWidth/dWidth);
				u32 d=m_pStatus->img_data[dx];

				screenBuf[((starty+y)*SCREEN_WIDTH)+startx+x]=
					(d&0x000000ff)<<16
					| (d&0x0000ff00)
					| (d&0x00ff0000)>>16
					| (d&0xff000000);
			}
		}
		/*for(int y=0; y<m_pStatus->img_height; y++)
		{
			for(int x=0; x<m_pStatus->img_width; x++)
			{
				u32 d=m_pStatus->img_data[(y*m_pStatus->img_width)+x];
				screenBuf[((starty+y)*SCREEN_WIDTH)+startx+x]=
					(d&0x000000ff)<<16
					| (d&0x0000ff00)
					| (d&0x00ff0000)>>16
					| (d&0xff000000);
			}
		}*/
	}

	if(doScreenshot)
	{
		//m_pFS->writeFile((const u8 *)m_fb.GetBuffer(), (1280*720*4), "/screenshot.raw");
		m_pFile->writeFile((const u8 *)m_fb.GetBuffer(), (1280*720*4), "/screenshot.raw");
		doScreenshot=FALSE;
	}

	m_fb.UpdateDisplay();

	if(frameCount++%10==0) {
		u64 timeNow=m_pTimer->GetClockTicks64();
		fps=1000000.0*10/(timeNow-lastUpdate);
		lastUpdate=timeNow;
	}
}

void Display::updateDir ()
{
	if(dirList!=NULL) delete[] dirList;
	dirList=new FILINFO[MAX_DIR_ENTRIES];
	//dirNum=m_pFS->readDir(dirList, MAX_DIR_ENTRIES, currentDir);
	dirNum=m_pFile->readDir(dirList, MAX_DIR_ENTRIES, currentDir);
	cursor=0;
}

void Display::selectDirEntry ()
{
	if(dirList!=NULL)
	{
		if(dirList[cursor].fattrib&AM_DIR)
		{
			if(currentDir[currentDir.GetLength()-1]!='/') currentDir+="/";
			currentDir+=dirList[cursor].fname;
			dirChanged=TRUE;
		} else {
			m_pStatus->currentTap=currentDir;
			m_pStatus->currentTap+="/";
			m_pStatus->currentTap+=dirList[cursor].fname;
			fileChanged=TRUE;
		}
	}
}

void Display::touchEventHandler (TTouchScreenEvent Event, unsigned nId, unsigned nPosX, unsigned nPosY)
{
	tPosX=nPosX;
	tPosY=nPosY;
}

void Display::touchEventStub (TTouchScreenEvent Event, unsigned nId, unsigned nPosX, unsigned nPosY)
{
	assert (s_pThis != 0);
	s_pThis->touchEventHandler (Event, nId, nPosX, nPosY);
}

void Display::mouseEventHandler (TMouseEvent Event, unsigned nButtons, unsigned nPosX, unsigned nPosY, int nWheelMove)
{
	if(!mShowCursor)
	{
		m_pMouse->ShowCursor(TRUE);
		mShowCursor=TRUE;
	}
	m_pMouse->SetCursor(nPosX, nPosY);

	mPosX=nPosX;
	mPosY=nPosY;
	mButtons=nButtons;

	switch(Event)
	{
		case MouseEventMouseWheel:
			cursor-=nWheelMove;
			if(cursor<0) cursor=0;
			if(cursor>=dirNum) cursor=dirNum-1;
			break;
		default:	
			break;
	}
}

void Display::mouseEventStub (TMouseEvent Event, unsigned nButtons, unsigned nPosX, unsigned nPosY, int nWheelMove)
{
	assert (s_pThis != 0);
	s_pThis->mouseEventHandler (Event, nButtons, nPosX, nPosY, nWheelMove);
}

void Display::navBack ()
{
	for(int i=currentDir.GetLength(); i>=0; i--)
	{
		if(currentDir[i]=='/')
		{
			char d[255];
			strncpy(d, currentDir, i);
			d[i]='\0';
			currentDir=d;
			break;
		}
	}
	dirChanged=TRUE;
	//updateLCD=TRUE;
}

void Display::keyHandler(const char *pString)
{
	assert (s_pThis != 0);

	lastKey=pString;

	// Up 
	if(!strcmp(pString, "\x1b[A"))
	{
		if(cursor>0)
		{
			cursor--;
			updateLCD=TRUE;
		}
	}
	// Down
	if(!strcmp(pString, "\x1b[B"))
	{
		if(cursor<(dirNum-1))
		{
			cursor++;
			updateLCD=TRUE;
		}
	}
	// Left
	if(!strcmp(pString, "\x1b[D"))
	{
		cursor-=10;
		if(cursor<0) cursor=0;
		updateLCD=TRUE;
	}
	// Right
	if(!strcmp(pString, "\x1b[C"))
	{
		cursor+=10;
		if(cursor>=dirNum) cursor=dirNum-1;
		updateLCD=TRUE;
	}
	// Backspace
	if(!strcmp(pString, "\177") || !strcmp(pString, "\x1b[1~"))
	{
		s_pThis->navBack();
	}
	// Enter
	if(!strcmp(pString, "\n"))
	{
		s_pThis->selectDirEntry();
	}
	// *
	if(!strcmp(pString, "*"))
	{
		if(!s_pThis->m_pTap->isPlaying())
		{
			s_pThis->m_pTap->playTap();
			//s_pThis->m_pTapeCart->setMode(MODE_STREAM);
		} else {
			if(s_pThis->m_pTap->isRecording())
			{
				s_pThis->m_pTap->stopTap();
				recordStopped=TRUE;
			} else {
				s_pThis->m_pTap->stopTap();
			}
		}
	}
	// F1
	if(!strcmp(pString, "\x1b[[A"))
	{
		s_pThis->m_pTap->playTap();
		//s_pThis->m_pTapeCart->setMode(MODE_STREAM);
	}
	// F2
	if(!strcmp(pString, "\x1b[[B"))
	{
		if(s_pThis->m_pTap->isRecording())
		{
			s_pThis->m_pTap->stopTap();
			recordStopped=TRUE;
		} else {
			s_pThis->m_pTap->stopTap();
		}
	}
	// F4
	if(!strcmp(pString, "\x1b[[D"))
	{
		s_pThis->m_pTap->recordTap((char *)s_pThis->m_pStatus->tap_buffer, TAP_MAX_SIZE);
	}
	// F5
	if(!strcmp(pString, "\x1b[[E"))
	{
		currentDir="/";
		dirChanged=TRUE;
	}
	// F6
	if(!strcmp(pString, "\x1b[17~"))
	{
		doScreenshot=TRUE;
	}
	// F8
	if(!strcmp(pString, "\x1b[19~"))
	{
		s_pThis->m_pStatus->rebootRequest=TRUE;
	}
	// +
	if(!strcmp(pString, "+") || !strcmp(pString, "\x27"))
	{
		int c=s_pThis->m_pTap->getCounterSeconds();
		s_pThis->m_pTap->seekSeconds(c+5);
	}
	// -
	if(!strcmp(pString, "-") || !strcmp(pString, "\xdf"))
	{
		unsigned c=s_pThis->m_pTap->getCounterSeconds();
		if(c<5) c=5;
		s_pThis->m_pTap->seekSeconds(c-5);
	}
	// /
	if(!strcmp(pString, "/"))
	{
		s_pThis->m_pStatus->currentTap="/pitap/browser.prg";
		fileChanged=TRUE;
	}
}

boolean Display::Initialize (void)
{
	m_fb.Initialize();
	pCDisplay=m_fb.GetDisplay();

	if (m_pKeyb == 0)
	{
		m_pKeyb = (CUSBKeyboardDevice *) m_pDeviceNameService->GetDevice ("ukbd1", FALSE);
		if (m_pKeyb != 0)
		{
			m_pKeyb->RegisterKeyPressedHandler(keyHandler);
		}
	}
	/*if (m_pMouse == 0)
	{
		m_pMouse = (CMouseDevice *) m_pDeviceNameService->GetDevice ("mouse1", FALSE);
		if (m_pMouse != 0)
		{
			m_pMouse->Setup(pCDisplay, TRUE);
			m_pMouse->RegisterEventHandler(mouseEventStub);
		}
	}*/
	//if (m_pTouchScreen == 0)
	//{
		//m_pTouchScreen = (CTouchScreenDevice *) m_pDeviceNameService->GetDevice ("touch1", FALSE);
		//if (m_pTouchScreen != 0)
		//{
			//m_pTouchScreen->Setup(pCDisplay);
			//m_pTouchScreen->RegisterEventHandler(touchEventStub);
			//const unsigned Coords[4]={0, SCREEN_WIDTH-1, 0, SCREEN_HEIGHT-1};
			//m_pTouchScreen->SetCalibration(Coords);
		//}
	//}

	currentDir="/";
	dirList=NULL;
	dirNum=0;
	cursor=0;
	updateDir();

	//for(int i=0; i<SCREEN_HEIGHT; i++) lastPulses[i]=0xffff;
	for(int i=0; i<SCREEN_HEIGHT; i++) lastPulses[i]=0x0;

	return TRUE;
}

