//
// tap.cpp
//
// Circle - A C++ bare metal environment for Raspberry Pi
// Copyright (C) 2015  R. Stange <rsta2@o2online.de>
// 
// PiTap (C) 2025 Mike Dawson https://gp2x.org/pitap
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//


#include <string.h>

#include "tap.h"

//#include <circle/gpiopin.h>
//#include <circle/spinlock.h>

//#define TAP_HDR 0x00
//#define TAP_VER 0x0c
//#define TAP_PLT 0x0d
//#define TAP_VID 0x0e
//#define TAP_RES 0x0f
//#define TAP_SIZ 0x10
//#define TAP_DAT 0x14

//CGPIOPin GPIOMotor;
//CGPIOPin GPIORead;
//CGPIOPin GPIOWrite;
//CGPIOPin GPIOSense;

//CGPIOPin GPIOLEDMotor;
//CGPIOPin GPIOLEDRecord;
//CGPIOPin GPIOLEDPlay;

//CSpinLock tapControlLock;

enum Platform {
	C64	= 0,
	VIC20	= 1,
	C16	= 2,
	PET	= 3,
	C5x0	= 4,
	C6x0	= 5
};

enum Video {
	PAL	= 0,
	NTSC	= 1,
	NTSCOLD	= 2,
	PALN	= 4
};

u8 *tap_data;
unsigned buf_length;
unsigned tap_length;
unsigned tap_pos;
unsigned tap_ver;

double clockSpeed=985248;
double clockSpeedDiv;
u64 counter_us;
u64 last_pulse_length;
unsigned missed_pulses;

boolean playing, recording;
boolean start_playing, start_recording;

boolean motor_on;
boolean motor_on_delayed_off;
boolean motor_led_on;
//u64 motor_on_time;
u64 motor_off_time;
#define MOTOR_DELAY_OFF_TIME 250000

Tap::Tap (CTimer *pTimer)
	:
	m_pTimer (pTimer)
{
}

Tap::~Tap (void) { }

boolean Tap::Initialize (void)
{
	GPIOMotor.AssignPin(TAP_PIN_MOTOR);
	GPIOMotor.SetMode(GPIOModeInputPullUp);

	GPIORead.AssignPin(TAP_PIN_READ);
	GPIORead.SetMode(GPIOModeOutput);
	GPIORead.Write(LOW);

	GPIOWrite.AssignPin(TAP_PIN_WRITE);
	GPIOWrite.SetMode(GPIOModeInput);

	GPIOSense.AssignPin(TAP_PIN_SENSE);
	GPIOSense.SetMode(GPIOModeOutput);
	GPIOSense.Write(HIGH);

	GPIOLEDMotor.AssignPin(TAP_PIN_LED_MOTOR);
	GPIOLEDMotor.SetMode(GPIOModeOutput);

	GPIOLEDRecord.AssignPin(TAP_PIN_LED_RECORD);
	GPIOLEDRecord.SetMode(GPIOModeOutput);

	GPIOLEDPlay.AssignPin(TAP_PIN_LED_PLAY);
	GPIOLEDPlay.SetMode(GPIOModeOutput);

	tap_data=NULL;
	buf_length=0;
	tap_length=0;
	tap_pos=0;

	return TRUE;
}

void Tap::hwInterface (void)
{
	static u64 last_pulse;
	static u64 next_pulse;
	static unsigned last_record_value;

	tapControlLock.Acquire();

	if(playing && tap_pos>=tap_length)
	{
		if(optionAutoRewind)
		{
			tap_pos=TAP_DAT;
			counter_us=0;
			if(optionAutoStop) playing=FALSE;
		} else playing=FALSE;
	}

	if(recording && tap_pos>=buf_length)
	{
			recording=FALSE;
			setSizeHeader();
	}

	if(GPIOMotor.Read()) // motor is off
	{
		if(motor_on)
		{
			motor_on=FALSE;
			motor_off_time=m_pTimer->GetClockTicks64();
		}
		if(motor_on_delayed_off)
		{
			if(m_pTimer->GetClockTicks64()>(motor_off_time+MOTOR_DELAY_OFF_TIME))
			{
				motor_on_delayed_off=FALSE;
				GPIOLEDMotor.Write(LOW);
				motor_led_on=FALSE;
			}
		}
		if((playing|recording) && !motor_on_delayed_off)
		{
			last_pulse=m_pTimer->GetClockTicks64();
		}
	} else { // motor is on
		if(!motor_on)
		{
			motor_on=TRUE;
			motor_on_delayed_off=TRUE;
			//motor_on_time=m_pTimer->GetClockTicks64();
		//}
		//if(!motor_led_on)
		//{
		//	if(m_pTimer->GetClockTicks64()-motor_on_time>50000)
		//	{
				GPIOLEDMotor.Write(HIGH);
				motor_led_on=TRUE;
		//	}
		}
	}

	if(!playing && !recording && sense_on)
	{
		GPIOSense.Write(HIGH);
		sense_on=FALSE;
		GPIOLEDRecord.Write(LOW);
		GPIOLEDPlay.Write(LOW);
	}

	if(start_playing) 
	{
		GPIOSense.Write(LOW);
		sense_on=TRUE;
		GPIOLEDPlay.Write(HIGH);
		start_playing=FALSE;
		last_pulse=m_pTimer->GetClockTicks64();
	}

	if(playing && motor_on_delayed_off)
	{
		u64 i=tap_data[tap_pos++];
		u64 pulse_length;

		if(i==0 && tap_ver==0) i=256;

		if(i==0 && tap_ver!=0 && tap_pos<=(tap_length-3)) {
			i=tap_data[tap_pos++];
			i|=(tap_data[tap_pos++]<<8);
			i|=(tap_data[tap_pos++]<<16);
			//pulse_length=i/0.985248;
			pulse_length=i*clockSpeedDiv;
		} else {
			//pulse_length=i/0.123156;
			pulse_length=i*8*clockSpeedDiv;
		}

		next_pulse=last_pulse+pulse_length;
		u64 time_now=m_pTimer->GetClockTicks64();

		if(next_pulse>time_now)
		{
			u64 delay=next_pulse-time_now;

			if(tap_ver==2)
			{
				m_pTimer->SimpleusDelay(delay);
				if(tap_pos&0x1) GPIORead.Write(HIGH);
				else GPIORead.Write(LOW);
			} else {
				m_pTimer->SimpleusDelay(delay/2);
				GPIORead.Write(HIGH);

				m_pTimer->SimpleusDelay(delay/2);
				GPIORead.Write(LOW);
			}
		} else {
			missed_pulses++;
		}

		last_pulse=m_pTimer->GetClockTicks64();
		counter_us+=pulse_length;
		last_pulse_length=pulse_length;
	}

	if(start_recording) 
	{
		GPIOSense.Write(LOW);
		sense_on=TRUE;
		GPIOLEDRecord.Write(HIGH);
		start_recording=FALSE;
		last_pulse=m_pTimer->GetClockTicks64();
	}

	if(recording && motor_on_delayed_off)
	{
		unsigned record_value=GPIOWrite.Read();
		if(record_value!=last_record_value)
		{
			last_record_value=record_value;
			if(record_value==HIGH)
			{
				u64 time_now=m_pTimer->GetClockTicks64();
				u64 pulse_length=time_now-last_pulse;

				//u64 i=pulse_length*0.123156;
				u64 i=pulse_length/(clockSpeedDiv*8);
				if(i>255 && tap_pos<=(buf_length-4))
				{
					//i=pulse_length*0.985248;
					i=pulse_length/clockSpeedDiv;
					tap_data[tap_pos++]=0;
					tap_data[tap_pos++]=(i&0x0000ff);
					tap_data[tap_pos++]=(i&0x00ff00)>>8;
					tap_data[tap_pos++]=(i&0xff0000)>>16;
				} else {
					tap_data[tap_pos++]=i;
				}

				last_pulse=time_now;
				counter_us+=pulse_length;
				last_pulse_length=pulse_length;
				if(tap_pos>tap_length) tap_length=tap_pos;
			}
		}
	}

	tapControlLock.Release();
}

unsigned Tap::getCounterSeconds()
{
	u64 c;
	tapControlLock.Acquire();
	c=counter_us;
	tapControlLock.Release();
	return c/1000000;
}

u64 Tap::getLastPulseLength()
{
	u64 i;
	//tapControlLock.Acquire();
	i=last_pulse_length;
	//tapControlLock.Release();
	return i;
}

unsigned Tap::getMissedPulses()
{
	u64 i;
	tapControlLock.Acquire();
	i=missed_pulses;
	tapControlLock.Release();
	return i;
}

boolean Tap::getMotor()
{
	//tapControlLock.Acquire();
	boolean m=motor_on;
	//tapControlLock.Release();
	return m;
}

unsigned Tap::getPos (void)
{
	tapControlLock.Acquire();
	unsigned pos=tap_pos;
	tapControlLock.Release();
	return pos;
}

unsigned Tap::getTapLength (void)
{
	tapControlLock.Acquire();
	unsigned len=pos2time(tap_length);
	tapControlLock.Release();
	return len;
}

unsigned Tap::getTapSize (void)
{
	tapControlLock.Acquire();
	unsigned len=tap_length;
	tapControlLock.Release();
	return len;
}

boolean Tap::isBusy()
{
	//tapControlLock.Acquire();
	boolean busy=playing|recording;
	//tapControlLock.Release();
	return busy;
}

boolean Tap::isPlaying()
{
	tapControlLock.Acquire();
	boolean p=playing;
	tapControlLock.Release();
	return p;
}

boolean Tap::isRecording()
{
	tapControlLock.Acquire();
	boolean r=recording;
	tapControlLock.Release();
	return r;
}

int Tap::mountTap (char *tap, const unsigned length)
{
	tapControlLock.Acquire();

	tap_data=(u8 *)tap;
	tap_length=length;
	buf_length=length;

	tap_pos=TAP_DAT;
	counter_us=0;
	missed_pulses=0;

	tap_ver=tap_data[TAP_VER];

	switch(tap_data[TAP_PLT]) {
		default:
		case C64:
			switch(tap_data[TAP_VID]) {
				default:
				case PAL:
					clockSpeed=985248;
					break;
				case NTSC:
					clockSpeed=1022730;
					break;
				case NTSCOLD:
					clockSpeed=1022730;
					break;
				case PALN:
					clockSpeed=1023440;
					break;
			}
			break;
		case VIC20:
			switch(tap_data[TAP_VID]) {
				default:
				case PAL:
					clockSpeed=1108405;
					break;
				case NTSC:
					clockSpeed=1022727;
					break;
			}
			break;
		case C16:
			switch(tap_data[TAP_VID]) {
				default:
				case PAL:
					clockSpeed=886724;
					break;
				case NTSC:
					clockSpeed=894886;
					break;
			}
			break;
		case PET:
			switch(tap_data[TAP_VID]) {
				default:
				case PAL:
					clockSpeed=1000000;
					break;
				case NTSC:
					clockSpeed=1000000;
					break;
			}
			break;
		case C5x0:
			switch(tap_data[TAP_VID]) {
				default:
				case PAL:
					clockSpeed=985248;
					break;
				case NTSC:
					clockSpeed=1022730;
					break;
			}
			break;
		case C6x0:
			switch(tap_data[TAP_VID]) {
				default:
				case PAL:
					clockSpeed=2000000;
					break;
				case NTSC:
					clockSpeed=2000000;
					break;
			}
			break;
	}
	clockSpeedDiv=1000000/clockSpeed;

	tapControlLock.Release();
	return 1;
}

int Tap::playTap ()
{
	tapControlLock.Acquire();

	playing=TRUE;
	start_playing=TRUE;
	recording=FALSE;

	tapControlLock.Release();
	return 1;
}

int Tap::recordTap (char *buffer, const unsigned length)
{
	tapControlLock.Acquire();

	tap_data=(u8 *)buffer;

	strcpy((char *)tap_data, "C64-TAPE-RAW");

	tap_data[TAP_VER]=1;

	tap_data[TAP_PLT]=C64;
	tap_data[TAP_VID]=PAL;
	tap_data[TAP_RES]=0;

	buf_length=length;
	tap_length=TAP_DAT;
	tap_pos=TAP_DAT;

	counter_us=0;
	missed_pulses=0;

	playing=FALSE;
	recording=TRUE;
	start_recording=TRUE;

	tapControlLock.Release();
	return 1;
}

int Tap::seekBytes (const unsigned pos)
{
	tapControlLock.Acquire();
	tap_pos=TAP_DAT+pos;
	counter_us=pos2time(tap_pos);
	tapControlLock.Release();
	return 1;
}

int Tap::seekSeconds (const unsigned time_s)
{
	u64 time_us=time_s*1000000;
	tapControlLock.Acquire();
	tap_pos=time2pos(time_us);
	counter_us=pos2time(tap_pos);
	tapControlLock.Release();
	return 1;
}

void Tap::setSizeHeader (void)
{
	unsigned tap_size=tap_pos-TAP_DAT;
	tap_data[TAP_SIZ]=tap_size&0xff;
	tap_data[TAP_SIZ+1]=(tap_size&0xff00)>>8;
	tap_data[TAP_SIZ+2]=(tap_size&0xff0000)>>16;
	tap_data[TAP_SIZ+3]=(tap_size&0xff000000)>>24;
}

void Tap::stopTap (void)
{
	tapControlLock.Acquire();
	playing=FALSE;
	if(recording)
	{
		recording=FALSE;
		setSizeHeader();
		tap_length=tap_pos;
	}
	tapControlLock.Release();
}

u64 Tap::pos2time(unsigned pos)
{
	u64 time=0;
	unsigned xpos;
	for(xpos=TAP_DAT; xpos<pos; xpos++) {
		if(xpos>=tap_length) break;
		unsigned i=(unsigned)tap_data[xpos];
		if(i!=0) {
			//time+=i/0.123156;
			time+=i*8*clockSpeedDiv;
		} else {
			unsigned d1=tap_data[xpos+1];
			unsigned d2=tap_data[xpos+2];
			unsigned d3=tap_data[xpos+3];
			//time+=((d3<<16)+(d2<<8)+d1)/0.985248;
			time+=((d3<<16)|(d2<<8)|d1)*clockSpeedDiv;
			xpos+=3;
		}
	}
	return time;
}

unsigned Tap::time2pos(u64 time_us)
{
	u64 xtime=0;
	unsigned xpos;
	for(xpos=TAP_DAT; xpos<tap_length; xpos++) {
		if(xtime>=time_us) break;
		int i=(int)tap_data[xpos];
		if(i!=0) {
			//xtime+=i/0.123156;
			xtime+=i*8*clockSpeedDiv;
		} else {
			int d1=tap_data[xpos+1];
			int d2=tap_data[xpos+2];
			int d3=tap_data[xpos+3];
			//xtime+=((d3<<16)+(d2<<8)+d1)/0.985248;
			xtime+=((d3<<16)|(d2<<8)|d1)*clockSpeedDiv;
			xpos+=3;
		}
	}
	return xpos;
}

