//
// fs.cpp
//
// Circle - A C++ bare metal environment for Raspberry Pi
// Copyright (C) 2015  R. Stange <rsta2@o2online.de>
// 
// PiTap (C) 2025 Mike Dawson https://gp2x.org/pitap
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//

#include <strings.h>

#include "fs.h"

#include "stb_image_config.h"
void abort(void) { while(1); }

//#define DRIVE		"SD:"

FS::FS (CInterruptSystem *pInterrupt, CTimer *pTimer, CActLED *pActLED, File *pFile, Status *pStatus,
		Tap *pTap, TapeCart *pTapeCart, TapUtils *pTapUtils)
	:
	m_pInterrupt (pInterrupt),
	m_pTimer (pTimer),
	m_pActLED (pActLED),
	m_pFile (pFile),
	m_pStatus (pStatus),
	m_pTap (pTap),
	m_pTapecart (pTapeCart),
	m_pTapUtils (pTapUtils)
{
}

FS::~FS (void)
{
}

boolean FS::Initialize (void)
{
	FRESULT fr;
	FILINFO fno;

	fr=f_stat("pitap", &fno);
	if(fr!=FR_OK) f_mkdir("/pitap");

	return TRUE;
}

/*int FS::readFile (u8 *data, const unsigned max_size, const char *filename)
{
	FIL fp;
	unsigned br;

	if (f_open(&fp, filename, FA_READ|FA_OPEN_EXISTING)!=FR_OK) return 0;
	if (f_read(&fp, data, max_size, &br)!=FR_OK) return 0;
	f_close(&fp);

	return br;
}*/

int FS::readFileIdx (u8 *data, const unsigned max_size, const char *filename)
{
	FIL fp;
	unsigned br=0;
	//*data='\0';

	const char *ext[2];
	ext[0]=".idx";
	ext[1]=".IDX";
	char *f=new char[255];
	for(int i=0; i<2; i++)
	{
		strncpy(f, filename, 255);
		strcpy(f+(strlen(f)-(strlen(ext[i]))), ext[i]);
		if (f_open(&fp, f, FA_READ|FA_OPEN_EXISTING)!=FR_OK) continue;
		if (f_read(&fp, data, max_size, &br)!=FR_OK) continue;
		f_close(&fp);
		break;
	}
	data[br++]='\0';
	return br;
}

void FS::freeImage(u32 *img_data)
{
	if(img_data!=0) stbi_image_free(img_data);
}

u32 *FS::readFileImage (u8 *data, const unsigned max_size, const char *filename,
		int *width, int *height, int *channels)
{
	FIL fp;
	unsigned br;

	const char *ext[4];
	ext[0]=".png";
	ext[1]=".PNG";
	ext[2]=".jpg";
	ext[3]=".JPG";
	char *f=new char[255];
	for(int i=0; i<4; i++)
	{
		strncpy(f, filename, 255);
		strcpy(f+(strlen(f)-(strlen(ext[i]))), ext[i]);
		if (f_open(&fp, f, FA_READ|FA_OPEN_EXISTING)!=FR_OK) continue;
		if (f_read(&fp, data, max_size, &br)!=FR_OK) continue;
		f_close(&fp);
		break;
	}
	return (u32 *)stbi_load_from_memory(data, br, width, height, channels, 4);
}

int FS::idxToArray (const char *src, idx_entry_t dest[])
{
	int count=0;

        for (const char *p = src; *p && count < 100; ) {
        /* skip spaces, tabs, newlines, and carriage returns */
        while (*p == ' ' || *p == '\t' || *p == '\n' || *p == '\r') ++p;
        if (!*p) break;

        if (*p == ';') {
            while (*p && *p != '\n' && *p != '\r') ++p;
            continue;
        }

        const char *nl = p;
        while (*nl && *nl != '\n' && *nl != '\r') ++nl;

        char line[512];
        size_t len = (size_t)(nl - p);
        if (len >= sizeof(line)) len = sizeof(line) - 1;
        memcpy(line, p, len);
        line[len] = '\0';

        char *space = strchr(line, ' ');
        if (!space) { p = nl + !!*nl; continue; }

        *space = '\0';
        const char *hex = line;
        const char *title = space + 1;
        while (*title == ' ' || *title == '-') ++title;

        dest[count].pos = 0;
        if (hex[0] == '0' && (hex[1] == 'x' || hex[1] == 'X'))
            hex += 2;
        for (; *hex; ++hex) {
            int digit;
            if (*hex >= '0' && *hex <= '9') digit = *hex - '0';
            else if (*hex >= 'a' && *hex <= 'f') digit = *hex - 'a' + 10;
            else if (*hex >= 'A' && *hex <= 'F') digit = *hex - 'A' + 10;
            else break;
            dest[count].pos = (dest[count].pos << 4) | digit;
        }
	dest[count].time=m_pTap->pos2time(dest[count].pos);

        strncpy(dest[count].title, title, sizeof(dest[count].title) - 1);
        dest[count].title[sizeof(dest[count].title) - 1] = '\0';
        ++count;

        p = nl;
        if (*p == '\r' && p[1] == '\n') p += 2;
        else if (*p == '\n' || *p == '\r') p += 1;
    }

    return count;
}

CString FS::idxToJson (idx_entry_t idxTable[], unsigned count)
{
	CString entry, entry2;

	entry="[\n";
	for (unsigned i=0; i<count; i++)
	{
		entry2.Format("{ \"time\": %d, \"description\": \"%s\" }", idxTable[i].time/1000000, idxTable[i].title);
		entry+=entry2;
		if(i<(count-1)) entry+=",\n";
		else entry+="\n";
	}
	entry+="]\n";
	return entry;
}

void FS::unmountFile (void)
{
	freeImage(m_pStatus->img_data);
	m_pStatus->img_data=0;

	m_pStatus->idx_buffer[0]='\0';
}

int FS::mountFile (const char *file)
{
	unmountFile();

	unsigned br=0;
	m_pStatus->currentTap=file;

	for(int i=strlen(file); i>=0; i--)
	{
		if(file[i]=='/') {
			strncpy(m_pStatus->currentTapShort, file+i+1, 255);
			break;
		}
	}

	const char *ext=file+strlen(file)-5;
	if(!strcasecmp(ext+1, ".d64"))
	{
		u8 *d64Buf=new u8[TAP_MAX_SIZE];
		br=m_pFile->readFile(d64Buf, TAP_MAX_SIZE, (const char *)m_pStatus->currentTap);

		// convert d64
		if(m_pStatus->optionConvertToTapecart)
		{
			m_pStatus->tapecart_size=m_pTapUtils->tapecartBrowserInit(m_pStatus->tapecart_buffer, m_pStatus->currentTapShort);
			m_pStatus->tapecart_size=m_pTapUtils->d642Tap (
					TCRT,
					d64Buf, br,
					m_pStatus->tapecart_buffer,
					m_pStatus->tapecart_size,
					m_pStatus->currentTapShort,
					m_pStatus->idx_buffer);
			// create tapecart loader tap
			m_pStatus->tap_size=m_pTapUtils->tapecartLoader2Tap (
					m_pStatus->tapecart_buffer, m_pStatus->tapecart_size,
					m_pStatus->tap_buffer, 0,
					m_pStatus->idx_buffer);
		} else {
			m_pStatus->tap_size=m_pTapUtils->d642Tap (
					//TURBOTAPE64_WITH_LOADER,
					m_pStatus->optionTapConversionFormat,
					d64Buf, br,
					m_pStatus->tap_buffer, 0,
					m_pStatus->currentTapShort,
					m_pStatus->idx_buffer);
			// tapecart to tap mode
			m_pTapUtils->mountTapecartTapLoader(m_pStatus->tapecart_buffer, (const char *)m_pStatus->currentTap);
			m_pStatus->tapecartDynamic=TRUE;
		}
		delete[] d64Buf;
		m_pTapecart->mountTapecart(m_pStatus->tapecart_buffer, m_pStatus->tapecart_size);
	} else if(!strcasecmp(ext+1, ".prg"))
	{
		u8 *prgBuf=new u8[TAP_MAX_SIZE];

		if(m_pStatus->optionConvertToTapecart)
		{
			// convert prg to tapecart
			br=m_pFile->readFile(prgBuf, TAP_MAX_SIZE, (const char *)m_pStatus->currentTap);

			m_pStatus->tapecart_size=m_pTapUtils->prg2Tapecart(
					prgBuf, br,
					m_pStatus->tapecart_buffer, TCRT_MAX_SIZE,
					m_pStatus->currentTapShort
					);

			m_pStatus->tapecartDynamic=TRUE;
			m_pTapecart->mountTapecart(m_pStatus->tapecart_buffer, m_pStatus->tapecart_size);

			// create tapecart loader tap
			m_pStatus->tap_size=m_pTapUtils->tapecartLoader2Tap (
					m_pStatus->tapecart_buffer, m_pStatus->tapecart_size,
					m_pStatus->tap_buffer, 0,
					m_pStatus->idx_buffer);
		} else {
			// convert prg to tap
			br=m_pFile->readFile(prgBuf, TAP_MAX_SIZE, (const char *)m_pStatus->currentTap);

			m_pStatus->tap_size=m_pTapUtils->prg2Tap (
					m_pStatus->optionTapConversionFormat,
					prgBuf, br,
					m_pStatus->tap_buffer, 0,
					m_pStatus->currentTapShort,
					m_pStatus->idx_buffer);


			// tapecart to tap mode
			m_pTapUtils->mountTapecartTapLoader(m_pStatus->tapecart_buffer, (const char *)m_pStatus->currentTap);
			m_pTapecart->mountTapecart(m_pStatus->tapecart_buffer, m_pStatus->tapecart_size);
		}

		delete[] prgBuf;
	} else if(!strcasecmp(ext+1, ".p00"))
	{
		u8 *p00Buf=new u8[TAP_MAX_SIZE];
		br=m_pFile->readFile(p00Buf, TAP_MAX_SIZE, (const char *)m_pStatus->currentTap);

		if(m_pStatus->optionConvertToTapecart)
		{
			// convert p00 to tapecart
			m_pStatus->tapecart_size=m_pTapUtils->p002Tap(
					TCRT,
					p00Buf, br,
					m_pStatus->tapecart_buffer, 0,
					m_pStatus->currentTapShort,
					m_pStatus->idx_buffer);

			m_pStatus->tapecartDynamic=TRUE;
			m_pTapecart->mountTapecart(m_pStatus->tapecart_buffer, m_pStatus->tapecart_size);

			// create tapecart loader tap
			m_pStatus->tap_size=m_pTapUtils->tapecartLoader2Tap(
					m_pStatus->tapecart_buffer, m_pStatus->tapecart_size,
					m_pStatus->tap_buffer, 0,
					m_pStatus->idx_buffer);

		} else {
			// convert p00 to tap
			m_pStatus->tap_size=m_pTapUtils->p002Tap (
					//TURBOTAPE64_WITH_LOADER,
					m_pStatus->optionTapConversionFormat,
					p00Buf, br,
					m_pStatus->tap_buffer, 0,
					m_pStatus->currentTapShort,
					m_pStatus->idx_buffer);

			// tapecart to tap mode
			m_pTapUtils->mountTapecartTapLoader(m_pStatus->tapecart_buffer, (const char *)m_pStatus->currentTap);
			m_pTapecart->mountTapecart(m_pStatus->tapecart_buffer, m_pStatus->tapecart_size);
		}

		delete[] p00Buf;
	} else if(!strcasecmp(ext+1, ".sid"))
	{
		CString path=file;
		int len=path.GetLength();
		char *wd=new char[len];
		int i;
		for(i=len; i>0 && path[i]!='/'; i--);
		strncpy(wd, (const char *)path, i);
		wd[i]=0;
		m_pTapecart->currentDir=wd;
		delete[] wd;

		u8 *prgBuf=new u8[TAP_MAX_SIZE];
		br=m_pFile->readFile(prgBuf, TAP_MAX_SIZE, "/pitap/sidplay64.prg");

		m_pStatus->tapecart_size=m_pTapUtils->prg2Tapecart(
				prgBuf, br,
				m_pStatus->tapecart_buffer, TCRT_MAX_SIZE,
				"SIDPLAY64"
				);
		m_pStatus->tapecartDynamic=TRUE;
		m_pTapecart->mountTapecart(m_pStatus->tapecart_buffer, m_pStatus->tapecart_size);
		delete[] prgBuf;

		// create tapecart loader tap
		m_pStatus->tap_size=m_pTapUtils->tapecartLoader2Tap (
				m_pStatus->tapecart_buffer, m_pStatus->tapecart_size,
				m_pStatus->tap_buffer, 0,
				m_pStatus->idx_buffer);
	} else if(!strcasecmp(ext+1, ".t64"))
	{
		u8 *t64Buf=new u8[TAP_MAX_SIZE];
		br=m_pFile->readFile(t64Buf, TAP_MAX_SIZE, (const char *)m_pStatus->currentTap);

		if(m_pStatus->optionConvertToTapecart)
		{
			// convert t64 to tapecart
			m_pStatus->tapecart_size=m_pTapUtils->tapecartBrowserInit(m_pStatus->tapecart_buffer, m_pStatus->currentTapShort);
			m_pStatus->tapecart_size=m_pTapUtils->t642Tap (
					TCRT,
					t64Buf, br,
					m_pStatus->tapecart_buffer,
					m_pStatus->tapecart_size,
					m_pStatus->currentTapShort,
					m_pStatus->idx_buffer);

			// create tapecart loader tap
			m_pStatus->tap_size=m_pTapUtils->tapecartLoader2Tap (
					m_pStatus->tapecart_buffer, m_pStatus->tapecart_size,
					m_pStatus->tap_buffer, 0,
					m_pStatus->idx_buffer);
		} else {
			// convert t64 to tap
			m_pStatus->tap_size=m_pTapUtils->t642Tap (
					//TURBOTAPE64_WITH_LOADER,
					m_pStatus->optionTapConversionFormat,
					t64Buf, br,
					m_pStatus->tap_buffer, 0,
					m_pStatus->currentTapShort,
					m_pStatus->idx_buffer);

			// tapecart to tap mode
			m_pTapUtils->mountTapecartTapLoader(m_pStatus->tapecart_buffer, (const char *)m_pStatus->currentTap);
			m_pTapecart->mountTapecart(m_pStatus->tapecart_buffer, m_pStatus->tapecart_size);
		}

		delete[] t64Buf;
	} else if(!strcasecmp(ext, ".tcrt"))
	{
		// load tapecart file
		m_pStatus->currentTapecart=m_pStatus->currentTap;
		br=m_pFile->readFile(m_pStatus->tapecart_buffer, TCRT_MAX_SIZE, (const char *)m_pStatus->currentTapecart);
		m_pStatus->tapecart_size=br;
		m_pStatus->tap_size=m_pTapUtils->tapecartLoader2Tap (
				m_pStatus->tapecart_buffer, m_pStatus->tapecart_size,
				m_pStatus->tap_buffer, 0,
				m_pStatus->idx_buffer);
		m_pStatus->tapecartDynamic=FALSE;
		m_pTapecart->mountTapecart(m_pStatus->tapecart_buffer, m_pStatus->tapecart_size);
	} else {
		// load tap file
		br=m_pFile->readFile(m_pStatus->tap_buffer, TAP_MAX_SIZE, (const char *)m_pStatus->currentTap);
		m_pStatus->tap_size=br;

		// tapecart to tap mode
		m_pTapUtils->mountTapecartTapLoader(m_pStatus->tapecart_buffer, (const char *)m_pStatus->currentTap);
		m_pTapecart->mountTapecart(m_pStatus->tapecart_buffer, m_pStatus->tapecart_size);

		// load idx file
		readFileIdx(m_pStatus->idx_buffer, IDX_MAX_SIZE,
				(const char *)m_pStatus->currentTap);
	}

	m_pTap->stopTap();
	m_pTap->mountTap((char *)m_pStatus->tap_buffer, m_pStatus->tap_size);
	m_pStatus->tap_length=m_pTap->getTapLength();

	// load image file
	m_pStatus->img_data=readFileImage(m_pStatus->img_buffer, IMG_MAX_SIZE,
			(const char *)m_pStatus->currentTap,
			&m_pStatus->img_width, &m_pStatus->img_height,
			&m_pStatus->img_channels);

	// update idx data
	m_pStatus->idxCount=idxToArray((const char *)m_pStatus->idx_buffer, m_pStatus->idxTable);
	m_pStatus->idxJson=idxToJson(m_pStatus->idxTable, m_pStatus->idxCount);

	m_pStatus->fileChanged=TRUE;

	return 1;
}

int FS::mountTapecart (const char *file)
{
	unsigned br=0;
	const char *ext=file+strlen(file)-5;

	m_pStatus->currentTapecart=file;
	for(int i=strlen(file); i>=0; i--)
	{
		if(file[i]=='/') {
			strncpy(m_pStatus->currentTapecartShort, file+i+1, 255);
			break;
		}
	}

	if(!strcasecmp(ext+1, ".sid"))
	{
		// load data file to tapecart
		u8 *dataBuf=new u8[TAP_MAX_SIZE];
		br=m_pFile->readFile(dataBuf, TAP_MAX_SIZE, (const char *)file);

		m_pStatus->tapecart_size=m_pTapUtils->data2Tapecart(
				dataBuf, br,
				m_pStatus->tapecart_buffer, TCRT_MAX_SIZE,
				m_pStatus->currentTapShort
				);
		delete[] dataBuf;

		//m_pStatus->currentTapecart=m_pStatus->currentTap;
		m_pTapecart->mountTapecart(m_pStatus->tapecart_buffer, m_pStatus->tapecart_size);
	} else {
		return mountFile(file);
	}

	return 1;
}

void FS::listRec(const char *path, FIL *fp_out)
{
    FRESULT res;
    DIR dir;
    FILINFO fno;
    //char next[FF_MAX_LFN + 1];
    CString next;

    /* Open directory */
    res = f_opendir(&dir, path);
    if (res != FR_OK) return;

    for (;;) {
        res = f_readdir(&dir, &fno);
        if (res != FR_OK || fno.fname[0] == 0) break;   /* Break on error or EOF */

        if (fno.fname[0] == '.' && (fno.fname[1] == 0 ||
                                   (fno.fname[1] == '.' && fno.fname[2] == 0)))
            continue; /* Skip "." and ".." */

        /* Build full path */
        //snprintf(next, sizeof(next), "%s/%s", path, fno.fname);
	next.Format("%s/%s", path, fno.fname);

        if (fno.fattrib & AM_DIR) {         /* Directory: recurse */
            listRec(next, fp_out);
        } else {                              /* File: print to index.txt */
            f_printf(fp_out, "%s\n", (const char *)next);
        }
    }

    f_closedir(&dir);
}

void FS::buildIndex(void)
{
    FRESULT res;
    FIL file;

    res = f_open(&file, "/pitap/pitap-index.txt", FA_CREATE_ALWAYS | FA_WRITE);
    if (res != FR_OK) return;

    listRec("", &file);   /* "" = root directory ("/") on the default drive */

    f_close(&file);
}

