//
// webserver.cpp
//
// Circle - A C++ bare metal environment for Raspberry Pi
// Copyright (C) 2015  R. Stange <rsta2@o2online.de>
//
// PiTap (C) 2025 Mike Dawson https://gp2x.org/pitap
// Parts from Pottendo-Pi1541 https://github.com/pottendo/pottendo-Pi1541
// 
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program.  If not, see <http://www.gnu.org/licenses/>.
//

#include <assert.h>
#include <circle/logger.h>
#include <circle/memory.h>
#include <circle/string.h>
#include <circle/util.h>
#include <stdio.h>

#include "webserver.h"

static const char index_app[] =
#include "web/pitap-html.h"
;

#define MAX_CONTENT_SIZE	10*1024*1024
#define MAX_DIR_ENTRIES		2048

static const char FromWebServer[] = "webserver";

CMemorySystem mem;

CString header="<html><head></head><body>"
	"<h3>PiTap v" PITAP_VER "</h3>"
	"&middot; "
	"<a href=\"/record\">Record</a> &middot; "
	"<a href=\"/play\">Play</a> &middot; "
	"<a href=\"/rewind\">Rewind/F.Fwd</a> &middot; "
	"<a href=\"/stop\">Stop</a> &middot; "
	"\n";

CString footer="<hr />"
	"&middot; "
	"<a href=\"/reboot\">Reboot</a> &middot; "
	"</body></html>";

CString statusHeader;


static char *extract_field(const char *field, const char *pPartHeader, char *filename, char *extension = nullptr)
{
	assert(pPartHeader != 0);
	char *startpos = strstr(pPartHeader, field);

	// special filename handling
	if (startpos != 0)
	{
		startpos += strlen(field);
		u16 fnl = 0;
		u16 exl = 0;
		u16 extensionStart = 0;
		while ((startpos[fnl] != '\"') && (fnl < 254))
		{
			filename[fnl] = startpos[fnl];
			if (filename[fnl] == '.')
				extensionStart = fnl + 1;
			fnl++;
		}
		filename[fnl] = '\0';
		if (!extension)
			return filename;
		while (extensionStart > 0 && extensionStart < fnl && exl < 9)
		{
			u8 tmp = filename[extensionStart + exl];
			if (tmp >= 65 && tmp <= 90)
				tmp += 32; // strtolower
			extension[exl] = tmp;
			exl++;
		}
		extension[exl] = '\0';
		//Kernel.log("found filename: '%s' %i", filename, strlen(filename));
		//Kernel.log("found extension: '%s' %i", extension, strlen(extension));
	}
	else
		filename[0] = '\0';
	return filename;
}

int percent_decode(char* out, const char* in)
{
    static const char tbl[256] = {
        -1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,-1,-1,-1,
        -1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,-1,-1,-1,
        -1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,-1,-1,-1,
         0, 1, 2, 3, 4, 5, 6, 7,  8, 9,-1,-1,-1,-1,-1,-1,
        -1,10,11,12,13,14,15,-1, -1,-1,-1,-1,-1,-1,-1,-1,
        -1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,-1,-1,-1,
        -1,10,11,12,13,14,15,-1, -1,-1,-1,-1,-1,-1,-1,-1,
        -1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,-1,-1,-1,
        -1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,-1,-1,-1,
        -1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,-1,-1,-1,
        -1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,-1,-1,-1,
        -1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,-1,-1,-1,
        -1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,-1,-1,-1,
        -1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,-1,-1,-1,
        -1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,-1,-1,-1,
        -1,-1,-1,-1,-1,-1,-1,-1, -1,-1,-1,-1,-1,-1,-1,-1
    };
    char c, v1, v2, *beg=out;
    if(in != NULL) {
        while((c=*in++) != '\0') {
            if(c == '%') {
                if((v1=tbl[(unsigned char)*in++])<0 || 
                   (v2=tbl[(unsigned char)*in++])<0) {
                    *beg = '\0';
                    return -1;
                }
                c = (v1<<4)|v2;
            }
            *out++ = c;
        }
    }
    *out = '\0';
    return 0;
}

CWebServer::CWebServer (CNetSubSystem *pNetSubSystem, CTimer *pTimer, CActLED *pActLED,
		File *pFile, FS *pFS, Tap *pTap, TapeCart *pTapeCart, Status *pStatus, CSocket *pSocket)
:	CHTTPDaemon (pNetSubSystem, pSocket, MAX_CONTENT_SIZE, 80, MAX_CONTENT_SIZE),
	m_pTimer (pTimer),
	m_pActLED (pActLED),
	m_pFile (pFile),
	m_pFS (pFS),
	m_pTap (pTap),
	m_pTapeCart (pTapeCart),
	m_pStatus (pStatus)
{
}

CWebServer::~CWebServer (void)
{
	m_pActLED = 0;
}

CHTTPDaemon *CWebServer::CreateWorker (CNetSubSystem *pNetSubSystem, CSocket *pSocket)
{
	return new CWebServer (pNetSubSystem, m_pTimer, m_pActLED, m_pFile, m_pFS, m_pTap, m_pTapeCart, m_pStatus, pSocket);
}

void CWebServer::setStatusHeader()
{
	const char *stat;

	if(m_pTap->isPlaying()) stat="playing";
	else if(m_pTap->isRecording()) stat="recording to /record.tap";
	else stat="stopped";

	statusHeader.Format("<p>Mounted tap file: %s</p>"
			"<p>Status: %s\n</p>"
			"<p>Counter: %d\n</p>"
			"<p>Temperature: %d, clock: %dmhz</p>",
			(const char *)m_pStatus->currentTap,
			stat,
			m_pTap->getCounterSeconds(),
			m_pStatus->cpuTemp,
			m_pStatus->clockRate/1000000
			);
}

THTTPStatus CWebServer::GetContent (const char  *pPath,
				    const char  *pParams,
				    const char  *pFormData,
				    u8	        *pBuffer,
				    unsigned    *pLength,
				    const char **ppContentType)
{
	assert (pPath != 0);
	assert (ppContentType != 0);
	assert (m_pActLED != 0);

	CString String;
	const u8 *pContent = 0;
	unsigned nLength = 0;

	if (   strcmp (pPath, "/") == 0
	    || strcmp (pPath, "/index.html") == 0)
	{
		String=index_app;
		pContent = (const u8 *) (const char *) String;
		nLength = String.GetLength ();
		*ppContentType = "text/html; utf-8";
	}
	else if (strcmp (pPath, "/status") == 0)
	{
		const char *stat="Stopped";
		if(m_pStatus->isPlaying) stat="Playing";
		else if(m_pStatus->isRecording) stat="Recording";

		const char *motor="Off";
		if(m_pStatus->isMotorOn) motor="On";

		const char *mode="Datasette";
		if(m_pTapeCart->tapecartMode==MODE_LOADER) mode="Tapecart loader";
		else if(m_pTapeCart->tapecartMode==MODE_C64COMMAND) mode="Tapecart command";

		String.Format("{"
				"\"mountedTapFile\": \"%s\","
				"\"status\": \"%s\","
				"\"motor\": \"%s\","
				"\"counter\": \"%03d\","
				"\"tapLength\": \"%d\","
				"\"temperature\": \"%dc\","
				"\"mem\": \"%d/%dmb free\","
				"\"version\": \"%s\","
				"\"mode\": \"%s\","
				"\"shiftreg\": \"%x\","
				"\"time\": \"%d\""
				"}\n",
				(const char *)m_pStatus->currentTap,
				stat, motor,
				m_pStatus->counterSeconds,
				m_pStatus->tap_length/1000000,
				m_pStatus->cpuTemp,
				mem.GetHeapFreeSpace(HEAP_ANY)/1000000,
				mem.GetMemSize()/1000000,
				PITAP_VER,
				mode,
				m_pTapeCart->shiftreg,
				//(const char *)m_pTapeCart->tcrtLog,
				m_pTimer->GetLocalTime()
				);

		pContent = (const u8 *) (const char *) String;
		nLength = String.GetLength ();
		*ppContentType = "application/json; charset=iso-8859-1";
	}
	else if (strcmp (pPath, "/config") == 0)
	{
		String=m_pStatus->getOptionsJson();
		pContent = (const u8 *) (const char *) String;
		nLength = String.GetLength ();
		*ppContentType = "application/json; charset=iso-8859-1";
	}
	else if (strcmp (pPath, "/configset") == 0)
	{
		char *args=new char[strlen(pParams)];
		percent_decode(args, pParams);
		m_pStatus->setOptions(args);
		delete[] args;
		String="{}";
		pContent = (const u8 *) (const char *) String;
		nLength = String.GetLength ();
		*ppContentType = "application/json; charset=iso-8859-1";
	}
	else if (strcmp (pPath, "/getidx") == 0)
	{
		String=m_pStatus->idxJson;
		if(String.GetLength()==0) String="[]\n";
		pContent = (const u8 *) (const char *) String;
		nLength = String.GetLength ();
		*ppContentType = "application/json; charset=iso-8859-1";
	}
	else if (strcmp (pPath, "/getlog") == 0)
	{
		String=m_pStatus->log;
		if(String.GetLength()==0) String="\n";
		pContent = (const u8 *) (const char *) String;
		nLength = String.GetLength ();
		*ppContentType = "text/html; charset=iso-8859-1";
	}
	else if (strcmp (pPath, "/download") == 0)
	{
		// file=...
		boolean error=FALSE;
		if(strlen(pParams)>5)
		{
			char *file=new char[strlen(pParams)];
			char *file_decoded=new char[strlen(pParams)];
			strcpy(file, pParams+5);
			percent_decode(file_decoded, file);
			u8 *data=new u8[TAP_MAX_SIZE];
			//unsigned br=m_pFS->readFile(data, TAP_MAX_SIZE, file_decoded);
			unsigned br=m_pFile->readFile(data, TAP_MAX_SIZE, file_decoded);
			if(br!=0) {
				pContent = (const u8 *) (const char *) data;
				nLength = br;
				*ppContentType = "application/octet-stream";
			} else error=TRUE;
			delete[] file;
			delete[] file_decoded;
			delete[] data;
		} else error=TRUE;
		if(error)
		{
			String="<p>Error opening file</p>";
			pContent = (const u8 *) (const char *) String;
			nLength = String.GetLength ();
			*ppContentType = "text/html; charset=iso-8859-1";
		}
	}
	else if (strcmp (pPath, "/browse") == 0)
	{
		// dir=...
		boolean error=FALSE;
		if(strlen(pParams)>4)
		{
			String="";
			char *ddir=new char[strlen(pParams)];
			char *ddir_decoded=new char[strlen(pParams)];
			strcpy(ddir, pParams+4);
			percent_decode(ddir_decoded, ddir);

			FILINFO *dirlist=new FILINFO[MAX_DIR_ENTRIES];
			//int num_entries=m_pFS->readDir(dirlist, MAX_DIR_ENTRIES, ddir_decoded);
			int num_entries=m_pFile->readDir(dirlist, MAX_DIR_ENTRIES, ddir_decoded);

			String+="[\n";
			for(int i=0; i<num_entries; i++) {
				String+="\"";
				String+=dirlist[i].fname;
				if(dirlist[i].fattrib&AM_DIR) String+="/";
				if((i+1)<num_entries) String+="\",\n";
				else String+="\"\n";
			}
			String+="]\n";

			delete[] dirlist;
			delete[] ddir;
			delete[] ddir_decoded;
			pContent = (const u8 *) (const char *) String;
			nLength = String.GetLength ();
			*ppContentType = "application/json; charset=iso-8859-1";
		} else error=TRUE;
		if(error)
		{
			String="<p>Error opening directory</p>";
			pContent = (const u8 *) (const char *) String;
			nLength = String.GetLength ();
			*ppContentType = "text/html; charset=iso-8859-1";
		}
	}
	else if (strcmp (pPath, "/mount") == 0)
	{
		// file=...
		String="";
		boolean error=FALSE;
		if(strlen(pParams)>5)
		{
			char *file=new char[strlen(pParams)];
			char *file_decoded=new char[strlen(pParams)];
			strcpy(file, pParams+5);
			percent_decode(file_decoded, file);

			m_pFS->mountFile(file_decoded);
			String.Format("{ \"msg\": \"Mounted %d bytes from TAP file %s\" }\n", m_pStatus->tap_size, (const char *)m_pStatus->currentTap);

			delete[] file;
			delete[] file_decoded;
		} else error=TRUE;
		if(error)
		{
			String+="{ \"msg\": \"error mounting file\" }\n";
		}
		pContent = (const u8 *) (const char *) String;
		nLength = String.GetLength ();
		*ppContentType = "application/json; charset=iso-8859-1";
	}
	else if (strcmp (pPath, "/play") == 0)
	{
		m_pTap->playTap();
		//m_pTapeCart->setMode(MODE_STREAM);
		String=header;
		setStatusHeader();
		String+=statusHeader;
		CString s;
		s.Format("<p>Playing %d bytes</p>\n", m_pStatus->tap_size);
		String+=s;
		String+=footer;
		pContent = (const u8 *) (const char *) String;
		nLength = String.GetLength ();
		*ppContentType = "text/html; charset=iso-8859-1";
	}
	else if (strcmp (pPath, "/stop") == 0)
	{
		boolean r=FALSE;
		if(m_pTap->isRecording()) r=TRUE;
		m_pTap->stopTap();

		if(r)
		{
			unsigned size=m_pTap->getTapSize();
			//m_pFS->writeFile(m_pStatus->tap_buffer, size, (const char *)m_pStatus->currentTap);
			m_pFile->writeFile(m_pStatus->tap_buffer, size, (const char *)m_pStatus->currentTap);
			m_pStatus->tap_length=m_pTap->getTapLength();
			String.Format("Stopping recording.  Writing data to %s\n", m_pStatus->currentTap);
		} else {
			String+="<p>Stopping playback</p>";
		}

		String+=footer;
		pContent = (const u8 *) (const char *) String;
		nLength = String.GetLength ();
		*ppContentType = "text/html; charset=iso-8859-1";
	}
	else if (strcmp (pPath, "/record") == 0)
	{
		m_pTap->recordTap((char *)m_pStatus->tap_buffer, TAP_MAX_SIZE);

		String=header;
		setStatusHeader();
		String+=statusHeader;
		String+=("<p>Recording</p>\n");
		String+=footer;
		pContent = (const u8 *) (const char *) String;
		nLength = String.GetLength ();
		*ppContentType = "text/html; charset=iso-8859-1";
	}
	else if (strcmp (pPath, "/rewind") == 0)
	{
		String=header;
		setStatusHeader();
		String+=statusHeader;

		String+="<form method=\"get\">"
			"<input type=\"text\" name=\"counter\">"
			"<input type=\"submit\" value=\"Set counter\">"
			"</form>";

		if(*pParams)
		{
			int count=atoi(strstr(pParams, "=")+1);
			m_pTap->seekSeconds(count);
			CString s;
			s.Format("<p>Setting tape counter to %d</p>\n", count);
			String+=s;
		}

		String+=footer;
		pContent = (const u8 *) (const char *) String;
		nLength = String.GetLength ();
		*ppContentType = "text/html; charset=iso-8859-1";
	}
	else if (strcmp (pPath, "/setcounter") == 0)
	{
		String="";
		if(*pParams)
		{
			int count=atoi(strstr(pParams, "=")+1);
			m_pTap->seekSeconds(count);
			CString s;
			s.Format("<p>Setting tape counter to %d</p>\n", count);
			String+=s;
		}
		pContent = (const u8 *) (const char *) String;
		nLength = String.GetLength ();
		*ppContentType = "text/html; charset=iso-8859-1";
	}
	else if (strcmp (pPath, "/reboot") == 0)
	{
		m_pStatus->rebootRequest=TRUE;
		String=header;
		setStatusHeader();
		String+=statusHeader;
		CString s;
		s.Format("<p>Rebooting</p>\n");
		String+=s;
		String+=footer;
		pContent = (const u8 *) (const char *) String;
		nLength = String.GetLength ();
		*ppContentType = "text/html; charset=iso-8859-1";
	}
	else if (strcmp (pPath, "/upload") == 0)
	{
		// dir=...
		char *dir=new char[strlen(pParams)];
		char *dir_decoded=new char[strlen(pParams)];
		strcpy(dir, strstr(pParams, "=")+1);
		percent_decode(dir_decoded, dir);

		const char *pPartHeader;
		const u8 *pPartData;
		unsigned nPartLength;
		char *filename=new char[255];
		String="";
		if (GetMultipartFormPart (&pPartHeader, &pPartData, &nPartLength))
		{
			assert (pPartHeader != 0);
			if (nPartLength > 0)
			{
				u8 *pfiledata = new u8[nPartLength];
				if (pfiledata != 0)
				{
					assert (pPartData != 0);
					memcpy (pfiledata, pPartData, nPartLength);

					extract_field("filename=\"", pPartHeader, filename);

					CString path;
					path.Format("%s/%s", dir_decoded, filename);
					String.Format("{ \"uploadPath\": \"%s\" }", (const char *)path);

					//m_pFS->writeFile(pfiledata, nPartLength, path);
					m_pFile->writeFile(pfiledata, nPartLength, path);
					delete[] pfiledata;
				}
			}
		}
		delete[] dir;
		delete[] filename;
		String+="\n";
		pContent = (const u8 *) (const char *) String;
		nLength = String.GetLength ();
		*ppContentType = "application/json; charset=iso-8859-1";
	}
	else if (strcmp (pPath, "/buildindex") == 0)
	{
		m_pFS->buildIndex();
		String="{ \"result\": \"index built\" }\n";
		pContent = (const u8 *) (const char *) String;
		nLength = String.GetLength ();
		*ppContentType = "application/json; charset=iso-8859-1";
	}
	else
	{
		return HTTPNotFound;
	}

	assert (pLength != 0);
	if (*pLength < nLength)
	{
		CLogger::Get ()->Write (FromWebServer, LogError, "Increase MAX_CONTENT_SIZE to at least %u", nLength);
		return HTTPInternalServerError;
	}

	assert (pBuffer != 0);
	assert (pContent != 0);
	assert (nLength > 0);
	memcpy (pBuffer, pContent, nLength);

	*pLength = nLength;

	return HTTPOK;
}

void CWebServer::WriteAccessLog(const CIPAddress &rRemoteIP, THTTPRequestMethod RequestMethod, const char *pRequestURI, THTTPStatus Status, unsigned nContentLength)
{
}

